// service_worker.js (Manifest V3) - GRAB v0.3
// Intercepts network requests to capture actual video URLs
// Stores captured media per tab for the content script to access

// Store captured media URLs per tab
const capturedMedia = new Map(); // tabId -> Map<url, {type, size, timestamp}>

// Video file patterns to capture
const VIDEO_PATTERNS = [
  /\.mp4(\?|$|#)/i,
  /\.webm(\?|$|#)/i,
  /\.m4v(\?|$|#)/i,
  /\.mov(\?|$|#)/i,
  /\.avi(\?|$|#)/i,
  /\.mkv(\?|$|#)/i,
  /\.flv(\?|$|#)/i,
  /\.ogv(\?|$|#)/i,
  /\.ts(\?|$|#)/i,      // MPEG-TS segments
  /video\/mp4/i,
  /video\/webm/i,
  /video\/x-flv/i,
  /video\/quicktime/i,
  /videoplayback/i,     // YouTube-style
  /googlevideo\.com/i,  // YouTube CDN
  /ytimg\.com.*\.webp/i, // Skip thumbnails
  /\.m3u8(\?|$|#)/i,    // HLS manifest
  /\.mpd(\?|$|#)/i,     // DASH manifest
  /mime=video/i,        // YouTube mime parameter
  /itag=\d+/i,          // YouTube itag parameter (video quality)
];

// URL patterns to SKIP (not actual videos)
const SKIP_PATTERNS = [
  /\.jpg(\?|$)/i,
  /\.jpeg(\?|$)/i,
  /\.png(\?|$)/i,
  /\.gif(\?|$)/i,
  /\.webp(\?|$)/i,
  /\.svg(\?|$)/i,
  /\.ico(\?|$)/i,
  /googleads/i,
  /doubleclick/i,
  /analytics/i,
  /tracking/i,
  /\.js(\?|$)/i,
  /\.css(\?|$)/i,
  /\/api\//i,
  /\/stats\//i,
];

// Content type patterns
const VIDEO_CONTENT_TYPES = [
  'video/mp4',
  'video/webm',
  'video/x-flv',
  'video/quicktime',
  'video/x-msvideo',
  'video/x-matroska',
  'video/mp2t',           // MPEG-TS
  'application/vnd.apple.mpegurl', // HLS
  'application/x-mpegurl',
  'application/dash+xml',
  'application/octet-stream', // Sometimes used for video
];

function shouldSkip(url) {
  for (const pattern of SKIP_PATTERNS) {
    if (pattern.test(url)) return true;
  }
  return false;
}

function isVideoUrl(url, contentType = '') {
  // Skip known non-video patterns
  if (shouldSkip(url)) return false;
  
  // Check content type first
  if (contentType) {
    const ct = contentType.toLowerCase();
    for (const vct of VIDEO_CONTENT_TYPES) {
      if (ct.includes(vct)) return true;
    }
    // Also check if content-type explicitly says video
    if (ct.startsWith('video/')) return true;
  }
  
  // Check URL patterns
  for (const pattern of VIDEO_PATTERNS) {
    if (pattern.test(url)) return true;
  }
  
  return false;
}

function getMediaType(url, contentType = '') {
  if (contentType.includes('mp4') || /\.mp4/i.test(url) || /mime=video%2Fmp4/i.test(url)) return 'MP4';
  if (contentType.includes('webm') || /\.webm/i.test(url) || /mime=video%2Fwebm/i.test(url)) return 'WebM';
  if (contentType.includes('mpegurl') || /\.m3u8/i.test(url)) return 'HLS';
  if (contentType.includes('dash') || /\.mpd/i.test(url)) return 'DASH';
  if (contentType.includes('mp2t') || /\.ts(\?|$)/i.test(url)) return 'TS';
  if (/googlevideo|videoplayback/i.test(url)) return 'Stream';
  if (/\.mov/i.test(url)) return 'MOV';
  return 'Video';
}

function formatSize(bytes) {
  if (!bytes || bytes === 0) return '';
  if (bytes < 1024) return bytes + ' B';
  if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(1) + ' KB';
  if (bytes < 1024 * 1024 * 1024) return (bytes / (1024 * 1024)).toFixed(1) + ' MB';
  return (bytes / (1024 * 1024 * 1024)).toFixed(2) + ' GB';
}

// Estimate duration based on file size and typical bitrates
function estimateDuration(bytes, type) {
  if (!bytes || bytes < 10000) return null;
  
  // Typical bitrates (bytes per second)
  const bitrates = {
    'TS': 500000,      // ~4 Mbps for TS segments
    'MP4': 625000,     // ~5 Mbps average
    'WebM': 500000,    // ~4 Mbps
    'Stream': 750000,  // ~6 Mbps for YouTube
    'Video': 625000,
  };
  
  const bitrate = bitrates[type] || 625000;
  const seconds = bytes / bitrate;
  
  if (seconds < 1) return '<1s';
  if (seconds < 60) return `~${Math.round(seconds)}s`;
  if (seconds < 3600) return `~${Math.floor(seconds / 60)}m ${Math.round(seconds % 60)}s`;
  return `~${Math.floor(seconds / 3600)}h ${Math.floor((seconds % 3600) / 60)}m`;
}

// Classify the capture: segment, chunk, or full video
function classifyCapture(bytes, type, url) {
  if (!bytes) return { label: 'Unknown', confidence: 'low' };
  
  // TS segments are typically 2-10 seconds each
  if (type === 'TS') {
    if (bytes < 2000000) return { label: 'Segment', confidence: 'high' };
    return { label: 'Large Segment', confidence: 'medium' };
  }
  
  // HLS/DASH manifests
  if (type === 'HLS' || type === 'DASH') {
    return { label: 'Manifest', confidence: 'high' };
  }
  
  // Size-based classification
  if (bytes < 500000) return { label: 'Tiny', confidence: 'high' };        // < 500KB
  if (bytes < 2000000) return { label: 'Segment', confidence: 'medium' };  // < 2MB
  if (bytes < 10000000) return { label: 'Clip', confidence: 'medium' };    // < 10MB
  if (bytes < 50000000) return { label: 'Short Video', confidence: 'medium' }; // < 50MB
  if (bytes < 200000000) return { label: 'Full Video', confidence: 'high' };   // < 200MB
  return { label: 'Large Video', confidence: 'high' };  // > 200MB
}

// Extract video quality from YouTube URLs
function getQualityFromUrl(url) {
  const itagMatch = url.match(/itag=(\d+)/);
  if (itagMatch) {
    const itag = parseInt(itagMatch[1]);
    // Common YouTube itags
    const qualities = {
      18: '360p', 22: '720p', 37: '1080p', 38: '3072p',
      137: '1080p', 136: '720p', 135: '480p', 134: '360p', 133: '240p',
      248: '1080p', 247: '720p', 244: '480p', 243: '360p',
      313: '2160p', 271: '1440p', 308: '1440p', 315: '2160p',
      // Audio-only
      140: 'audio', 141: 'audio', 251: 'audio', 250: 'audio', 249: 'audio',
    };
    return qualities[itag] || `itag:${itag}`;
  }
  return null;
}

// Listen for web requests to capture video URLs
chrome.webRequest.onHeadersReceived.addListener(
  (details) => {
    const { tabId, url, responseHeaders, type } = details;
    if (tabId < 0) return; // Ignore non-tab requests
    
    // Only look at media and xmlhttprequest types (where videos typically come from)
    if (!['media', 'xmlhttprequest', 'other'].includes(type)) return;
    
    // Get content type and size from headers
    let contentType = '';
    let contentLength = 0;
    
    for (const header of responseHeaders || []) {
      const name = header.name.toLowerCase();
      if (name === 'content-type') contentType = header.value || '';
      if (name === 'content-length') contentLength = parseInt(header.value || '0', 10);
    }
    
    // Check if this is a video
    if (!isVideoUrl(url, contentType)) return;
    
    // Skip tiny files (likely not full videos) - but keep manifests and streams
    const isManifest = /\.(m3u8|mpd)(\?|$)/i.test(url);
    const isStream = /googlevideo|videoplayback/i.test(url);
    if (!isManifest && !isStream && contentLength > 0 && contentLength < 50000) return; // < 50KB
    
    // Store the captured URL
    if (!capturedMedia.has(tabId)) {
      capturedMedia.set(tabId, new Map());
    }
    
    const tabMedia = capturedMedia.get(tabId);
    const mediaType = getMediaType(url, contentType);
    const quality = getQualityFromUrl(url);
    
    // Use a shorter key to avoid duplicate entries
    // For YouTube, use the itag as part of the key
    let urlKey = url.split('?')[0];
    if (/googlevideo|videoplayback/i.test(url)) {
      const itag = url.match(/itag=(\d+)/);
      urlKey = itag ? `youtube-${itag[1]}` : url.slice(0, 100);
    }
    
    // Skip if already captured with same key
    if (tabMedia.has(urlKey)) {
      // Update size if larger
      const existing = tabMedia.get(urlKey);
      if (contentLength > existing.size) {
        existing.size = contentLength;
        existing.sizeStr = formatSize(contentLength);
        existing.estDuration = estimateDuration(contentLength, existing.type);
        existing.classification = classifyCapture(contentLength, existing.type, url);
        existing.url = url; // Update to newer URL (may have fresher token)
      }
      return;
    }
    
    const classification = classifyCapture(contentLength, mediaType, url);
    const estDuration = estimateDuration(contentLength, mediaType);
    
    const mediaInfo = {
      url: url,
      type: mediaType,
      quality: quality,
      contentType,
      size: contentLength,
      sizeStr: formatSize(contentLength),
      estDuration: estDuration,
      classification: classification,
      timestamp: Date.now(),
      downloadable: !isManifest && /^https?:\/\//i.test(url) && quality !== 'audio',
      isAudio: quality === 'audio',
    };
    
    tabMedia.set(urlKey, mediaInfo);
    
    console.log(`[GRAB] Captured: ${mediaType}${quality ? ' ' + quality : ''} ${mediaInfo.sizeStr || ''} - ${url.slice(0, 100)}...`);
    
    // Notify content script
    chrome.tabs.sendMessage(tabId, {
      type: 'GRAB_MEDIA_CAPTURED',
      media: mediaInfo
    }).catch(() => {}); // Ignore if content script not ready
  },
  { urls: ['<all_urls>'] },
  ['responseHeaders']
);

// Clean up when tab closes
chrome.tabs.onRemoved.addListener((tabId) => {
  capturedMedia.delete(tabId);
});

// Handle messages from content script
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  const tabId = sender.tab?.id;
  
  if (msg.type === 'GRAB_GET_MEDIA') {
    // Return captured media for this tab
    const tabMedia = capturedMedia.get(tabId);
    if (!tabMedia || tabMedia.size === 0) {
      sendResponse({ media: [] });
      return true;
    }
    
    const mediaList = Array.from(tabMedia.values())
      .sort((a, b) => b.size - a.size); // Largest first
    
    sendResponse({ media: mediaList });
    return true;
  }
  
  if (msg.type === 'GRAB_DOWNLOAD') {
    const { url, filename, saveAs } = msg;
    
    console.log('[GRAB] Download request:', { url: url.slice(0, 100), filename });
    
    chrome.downloads.download(
      { url, filename, saveAs: saveAs || false },
      (downloadId) => {
        const err = chrome.runtime.lastError;
        if (err) {
          console.error('[GRAB] Download failed:', err.message);
          sendResponse({ ok: false, error: err.message });
        } else {
          console.log('[GRAB] Download started:', downloadId);
          sendResponse({ ok: true, downloadId });
        }
      }
    );
    
    return true; // Keep channel open for async response
  }
  
  if (msg.type === 'GRAB_CLEAR_MEDIA') {
    if (tabId) capturedMedia.delete(tabId);
    sendResponse({ ok: true });
    return true;
  }
});

// Handle extension icon click
chrome.action.onClicked.addListener((tab) => {
  chrome.tabs.sendMessage(tab.id, { type: 'GRAB_TOGGLE_PANEL' });
});

console.log('[GRAB] v0.3.1 Service worker initialized - intercepting video requests');
