// content.js - GRAB Video Download Overlay v0.3
// Shows a dropdown with captured URLs when clicking the GRAB button on videos
//
// v0.3: Dropdown menu on video overlay with all captured options
// - Works with YouTube, Vimeo, CNN, and most video sites
// - Shows captured network URLs directly on the video
// - Displays file sizes and types for easy selection

const overlays = new Map(); // videoEl -> { btn, dropdown }
let rafId = null;
let panelEl = null;
let capturedMedia = []; // Media captured from network requests
let lastCaptureTime = 0;

// ===== Toast Notifications =====
function toast(message, type = "info", title = null, ms = 2500) {
  const icons = {
    info: "💡",
    success: "✅",
    warning: "⚠️",
    error: "❌",
  };

  const el = document.createElement("div");
  el.className = `grab-toast grab-toast-${type}`;
  el.innerHTML = `
    <span class="grab-toast-icon">${icons[type] || icons.info}</span>
    <div class="grab-toast-content">
      ${title ? `<div class="grab-toast-title">${title}</div>` : ""}
      <div class="grab-toast-message">${message}</div>
    </div>
  `;
  document.documentElement.appendChild(el);

  // Stack toasts
  const existingToasts = document.querySelectorAll(".grab-toast");
  if (existingToasts.length > 1) {
    const offset = (existingToasts.length - 1) * 60;
    el.style.bottom = `${16 + offset}px`;
  }

  setTimeout(() => {
    el.classList.add("grab-toast-out");
    setTimeout(() => el.remove(), 150);
  }, ms);
}

// ===== URL & Filename Helpers =====
function sanitizeFilenamePart(s) {
  return (s || "")
    .replace(/[\\/:*?"<>|]+/g, "_")
    .replace(/\s+/g, " ")
    .trim()
    .slice(0, 80);
}

function guessExtFromUrl(url) {
  try {
    const u = new URL(url);
    const m = u.pathname.match(/\.([a-z0-9]{2,5})$/i);
    if (m) return m[1].toLowerCase();
  } catch {}
  return "mp4";
}

function pickBestVideoUrl(video) {
  // Prefer currentSrc when available (resolves <source> selection)
  const candidate = video.currentSrc || video.src || "";
  if (candidate) return candidate;

  // Fallback: look for <source src="...">
  const srcEl = video.querySelector("source[src]");
  return srcEl ? srcEl.src : "";
}

function getVideoType(url) {
  if (!url) return { type: "none", downloadable: false };
  if (url.startsWith("blob:")) return { type: "blob", downloadable: false };
  if (url.includes(".m3u8") || url.includes("manifest")) return { type: "hls", downloadable: false };
  if (url.includes(".mpd")) return { type: "dash", downloadable: false };
  if (/^https?:\/\//i.test(url)) return { type: "direct", downloadable: true };
  return { type: "unknown", downloadable: false };
}

function truncateUrl(url, maxLen = 40) {
  if (!url) return "";
  if (url.length <= maxLen) return url;
  return url.slice(0, maxLen - 3) + "...";
}

// ===== Fetch Captured Media =====
async function fetchCapturedMedia() {
  try {
    const resp = await chrome.runtime.sendMessage({ type: 'GRAB_GET_MEDIA' });
    capturedMedia = resp?.media || [];
    lastCaptureTime = Date.now();
    return capturedMedia;
  } catch (e) {
    console.error('[GRAB] Failed to fetch media:', e);
    return [];
  }
}

// ===== Download Request =====
async function requestDownload(url, statusCallback) {
  const title = sanitizeFilenamePart(document.title) || "video";
  const ts = new Date().toISOString().replace(/[:.]/g, "-").slice(0, 19);
  const ext = guessExtFromUrl(url);
  const filename = `GRAB/${title}__${ts}.${ext}`;

  statusCallback?.('downloading');

  return new Promise((resolve) => {
    chrome.runtime.sendMessage(
      { type: "GRAB_DOWNLOAD", url, filename, saveAs: false },
      (resp) => {
        if (!resp || !resp.ok) {
          statusCallback?.('error', resp?.error);
          resolve({ ok: false, error: resp?.error || "unknown" });
        } else {
          statusCallback?.('success');
          resolve({ ok: true, downloadId: resp.downloadId });
        }
      }
    );
  });
}

// ===== Create Dropdown Menu for Video =====
function createDropdownContent(video) {
  const url = pickBestVideoUrl(video);
  const { type, downloadable } = getVideoType(url);
  
  // Get video element duration if available
  const videoDuration = video.duration && !isNaN(video.duration) ? formatDuration(video.duration) : null;
  
  // Sort captured media: largest downloadable first, then by size
  const sorted = [...capturedMedia].sort((a, b) => {
    // Prioritize "Full Video" and "Large Video" classifications
    const classOrder = { 'Large Video': 0, 'Full Video': 1, 'Short Video': 2, 'Clip': 3, 'Segment': 4, 'Large Segment': 5, 'Tiny': 6, 'Manifest': 7 };
    const aOrder = classOrder[a.classification?.label] ?? 5;
    const bOrder = classOrder[b.classification?.label] ?? 5;
    if (aOrder !== bOrder) return aOrder - bOrder;
    
    // Then by downloadable
    if (a.downloadable && !b.downloadable) return -1;
    if (!a.downloadable && b.downloadable) return 1;
    
    // Then by size
    return (b.size || 0) - (a.size || 0);
  });
  
  // Take top 10 for the dropdown
  const topMedia = sorted.slice(0, 10);
  
  // Count by classification
  const fullVideos = capturedMedia.filter(m => ['Full Video', 'Large Video', 'Short Video'].includes(m.classification?.label)).length;
  const segments = capturedMedia.filter(m => ['Segment', 'Large Segment', 'Tiny'].includes(m.classification?.label)).length;
  
  let html = `
    <div class="grab-dropdown-header">
      <span class="grab-dropdown-title">📥 GRAB Downloads</span>
      <span class="grab-dropdown-count">${fullVideos > 0 ? `${fullVideos} video${fullVideos > 1 ? 's' : ''}` : `${capturedMedia.length} captured`}</span>
    </div>
  `;
  
  // If video element has direct URL
  if (downloadable && url) {
    html += `
      <div class="grab-dropdown-section">Direct Video URL</div>
      <div class="grab-dropdown-item grab-dropdown-direct" data-url="${encodeURIComponent(url)}">
        <span class="grab-dropdown-icon">🎬</span>
        <div class="grab-dropdown-info">
          <div class="grab-dropdown-type">DIRECT ${type.toUpperCase()}</div>
          <div class="grab-dropdown-meta">${videoDuration ? `⏱️ ${videoDuration}` : 'Duration unknown'}</div>
          <div class="grab-dropdown-url">${truncateUrl(url, 40)}</div>
        </div>
        <span class="grab-dropdown-action">⬇️</span>
      </div>
    `;
  }
  
  // Video element info (even if blob)
  if (!downloadable && videoDuration) {
    html += `
      <div class="grab-dropdown-section">Video Element Info</div>
      <div class="grab-dropdown-info-box">
        <div>⏱️ Duration: <strong>${videoDuration}</strong></div>
        <div>📺 Type: <strong>${type.toUpperCase()}</strong></div>
        ${type === 'blob' ? '<div class="grab-hint">💡 Look for matching duration in Captured Streams below</div>' : ''}
      </div>
    `;
  }
  
  // Captured network requests
  if (topMedia.length > 0) {
    const sectionLabel = fullVideos > 0 
      ? `Captured Streams (${fullVideos} likely video${fullVideos > 1 ? 's' : ''}, ${segments} segments)`
      : `Captured Streams (${capturedMedia.length})`;
    html += `<div class="grab-dropdown-section">${sectionLabel}</div>`;
    
    for (const m of topMedia) {
      const qualityBadge = m.quality ? `<span class="grab-quality">${m.quality}</span>` : '';
      const sizeBadge = m.sizeStr ? `<span class="grab-size">${m.sizeStr}</span>` : '';
      const durationBadge = m.estDuration ? `<span class="grab-duration">⏱️${m.estDuration}</span>` : '';
      const classLabel = m.classification?.label || '';
      const classClass = classLabel.toLowerCase().replace(/\s+/g, '-');
      const isLikelyFull = ['Full Video', 'Large Video', 'Short Video'].includes(classLabel);
      const statusClass = m.isAudio ? 'audio' : (m.downloadable ? (isLikelyFull ? 'downloadable likely-full' : 'downloadable') : 'stream');
      
      html += `
        <div class="grab-dropdown-item ${statusClass}" data-url="${encodeURIComponent(m.url)}" data-downloadable="${m.downloadable && !m.isAudio}">
          <span class="grab-dropdown-icon">${m.isAudio ? '🔊' : (isLikelyFull ? '🎬' : (m.downloadable ? '📦' : '📡'))}</span>
          <div class="grab-dropdown-info">
            <div class="grab-dropdown-type">
              ${m.type} ${qualityBadge}
              ${classLabel ? `<span class="grab-class grab-class-${classClass}">${classLabel}</span>` : ''}
            </div>
            <div class="grab-dropdown-meta">${sizeBadge} ${durationBadge}</div>
            <div class="grab-dropdown-url">${truncateUrl(m.url, 40)}</div>
          </div>
          <span class="grab-dropdown-action">${m.isAudio ? '🔊' : (m.downloadable ? '⬇️' : '🔗')}</span>
        </div>
      `;
    }
    
    if (capturedMedia.length > 10) {
      html += `<div class="grab-dropdown-more">... and ${capturedMedia.length - 10} more (open Full Panel)</div>`;
    }
  } else {
    html += `
      <div class="grab-dropdown-empty">
        <div>No streams captured yet</div>
        <small>Play the video to capture URLs</small>
      </div>
    `;
  }
  
  // Show blob warning if needed
  if (type === 'blob' && topMedia.length === 0) {
    html += `
      <div class="grab-dropdown-warning">
        ⚠️ Blob URL detected. Play video to capture actual stream URLs.
      </div>
    `;
  }
  
  // Footer with actions
  html += `
    <div class="grab-dropdown-footer">
      <button class="grab-dropdown-btn grab-refresh-btn">🔄 Refresh</button>
      <button class="grab-dropdown-btn grab-panel-btn">📊 Full Panel</button>
    </div>
  `;
  
  return html;
}

// Format duration in seconds to readable format
function formatDuration(seconds) {
  if (!seconds || isNaN(seconds)) return null;
  const h = Math.floor(seconds / 3600);
  const m = Math.floor((seconds % 3600) / 60);
  const s = Math.floor(seconds % 60);
  if (h > 0) return `${h}:${m.toString().padStart(2, '0')}:${s.toString().padStart(2, '0')}`;
  return `${m}:${s.toString().padStart(2, '0')}`;
}

// ===== Overlay Button & Dropdown =====
function ensureOverlayForVideo(video) {
  if (overlays.has(video)) return;

  // Create container
  const container = document.createElement("div");
  container.className = "grab-overlay-container";
  
  // Create button
  const btn = document.createElement("div");
  btn.className = "grab-overlay-btn";
  btn.innerHTML = `<span class="grab-icon">⬇️</span> GRAB`;
  btn.title = "Click to see download options";
  
  // Create dropdown
  const dropdown = document.createElement("div");
  dropdown.className = "grab-dropdown";
  dropdown.style.display = "none";
  
  container.appendChild(btn);
  container.appendChild(dropdown);
  document.documentElement.appendChild(container);
  
  let dropdownOpen = false;
  
  // Toggle dropdown on click
  btn.addEventListener("click", async (e) => {
    e.preventDefault();
    e.stopPropagation();
    
    if (dropdownOpen) {
      dropdown.style.display = "none";
      dropdownOpen = false;
      return;
    }
    
    // Fetch latest captured media
    await fetchCapturedMedia();
    
    // Update dropdown content
    dropdown.innerHTML = createDropdownContent(video);
    dropdown.style.display = "block";
    dropdownOpen = true;
    
    // Attach event listeners to dropdown items
    dropdown.querySelectorAll(".grab-dropdown-item").forEach(item => {
      item.addEventListener("click", async (e) => {
        e.stopPropagation();
        const url = decodeURIComponent(item.dataset.url);
        const isDownloadable = item.dataset.downloadable === "true";
        
        if (!isDownloadable) {
          // Copy URL to clipboard for non-downloadable items
          try {
            await navigator.clipboard.writeText(url);
            toast("URL copied to clipboard!", "info", "GRAB");
          } catch {
            toast("Couldn't copy URL", "warning");
          }
          return;
        }
        
        const actionEl = item.querySelector(".grab-dropdown-action");
        const origAction = actionEl.textContent;
        
        const result = await requestDownload(url, (status, error) => {
          if (status === 'downloading') actionEl.textContent = '⏳';
          else if (status === 'success') actionEl.textContent = '✅';
          else if (status === 'error') {
            actionEl.textContent = '❌';
            toast(`Download failed: ${error}`, "error", "Error", 4000);
          }
        });
        
        if (result.ok) {
          toast("Download started!", "success", "GRAB");
        }
        
        setTimeout(() => {
          actionEl.textContent = origAction;
        }, 2000);
      });
    });
    
    // Refresh button
    dropdown.querySelector(".grab-refresh-btn")?.addEventListener("click", async (e) => {
      e.stopPropagation();
      await fetchCapturedMedia();
      dropdown.innerHTML = createDropdownContent(video);
      toast(`Found ${capturedMedia.length} streams`, "info");
    });
    
    // Full panel button
    dropdown.querySelector(".grab-panel-btn")?.addEventListener("click", (e) => {
      e.stopPropagation();
      dropdown.style.display = "none";
      dropdownOpen = false;
      createPanel();
    });
  });
  
  // Close dropdown when clicking outside
  document.addEventListener("click", (e) => {
    if (dropdownOpen && !container.contains(e.target)) {
      dropdown.style.display = "none";
      dropdownOpen = false;
    }
  });
  
  overlays.set(video, { btn, dropdown, container });
  schedulePositionLoop();
}

// ===== Position Update Loop =====
function updateOverlayPositions() {
  for (const [video, overlay] of overlays.entries()) {
    const { container, btn } = overlay;
    
    if (!video.isConnected) {
      container.remove();
      overlays.delete(video);
      continue;
    }

    const rect = video.getBoundingClientRect();

    // Hide if offscreen
    const offscreen =
      rect.bottom < 0 ||
      rect.top > window.innerHeight ||
      rect.right < 0 ||
      rect.left > window.innerWidth;

    // Hide if too small
    const tooSmall = rect.width < 120 || rect.height < 90;

    if (offscreen || tooSmall) {
      container.style.display = "none";
      continue;
    }

    container.style.display = "block";

    // Position at top-right of video
    const padding = 10;
    const btnWidth = 80;
    const x = Math.min(window.innerWidth - btnWidth - 8, rect.right - padding - btnWidth);
    const y = Math.max(8, rect.top + padding);

    container.style.left = `${Math.round(x)}px`;
    container.style.top = `${Math.round(y)}px`;

    // More visible when hovering video
    const isHover = video.matches(":hover");
    btn.style.opacity = isHover ? "1" : "";
  }

  rafId = requestAnimationFrame(updateOverlayPositions);
}

function schedulePositionLoop() {
  if (rafId != null) return;
  rafId = requestAnimationFrame(updateOverlayPositions);
}

// ===== Video Scanning =====
function scanForVideos(root = document) {
  const vids = root.querySelectorAll ? root.querySelectorAll("video") : [];
  vids.forEach(ensureOverlayForVideo);
}

// ===== Status Panel (v0.2 with Captured Media) =====
async function createPanel() {
  if (panelEl) {
    panelEl.remove();
    panelEl = null;
    return;
  }

  // Get captured media from background
  try {
    const resp = await chrome.runtime.sendMessage({ type: 'GRAB_GET_MEDIA' });
    capturedMedia = resp?.media || [];
  } catch (e) {
    capturedMedia = [];
  }

  const videos = Array.from(document.querySelectorAll("video"));
  const videoData = videos.map((v) => {
    const url = pickBestVideoUrl(v);
    const { type, downloadable } = getVideoType(url);
    return { video: v, url, type, downloadable };
  });

  const downloadableCount = videoData.filter((v) => v.downloadable).length;
  const capturedDownloadable = capturedMedia.filter(m => m.downloadable).length;

  panelEl = document.createElement("div");
  panelEl.className = "grab-panel";
  panelEl.innerHTML = `
    <div class="grab-panel-header">
      <div class="grab-panel-title">
        <span class="grab-logo">📥</span>
        GRAB <span class="grab-version">v0.2</span>
      </div>
      <button class="grab-panel-close">✕</button>
    </div>
    <div class="grab-panel-body">
      <div class="grab-panel-stats">
        <div class="grab-panel-stat">
          <div class="grab-panel-stat-value">${capturedMedia.length}</div>
          <div class="grab-panel-stat-label">Captured URLs</div>
        </div>
        <div class="grab-panel-stat">
          <div class="grab-panel-stat-value">${capturedDownloadable}</div>
          <div class="grab-panel-stat-label">Downloadable</div>
        </div>
        <div class="grab-panel-stat">
          <div class="grab-panel-stat-value">${videos.length}</div>
          <div class="grab-panel-stat-label">Video Tags</div>
        </div>
      </div>
      
      ${capturedMedia.length > 0 ? `
        <div class="grab-panel-section">
          <div class="grab-panel-section-title">🎯 Captured Network Requests (${capturedMedia.length})</div>
          <div class="grab-panel-captured">
            ${capturedMedia.map((m, i) => `
              <div class="grab-panel-media ${m.downloadable ? 'downloadable' : 'stream'}" data-captured-index="${i}">
                <div class="grab-panel-media-type">${m.type}${m.quality ? `<br><small>${m.quality}</small>` : ''}</div>
                <div class="grab-panel-media-info">
                  <div class="grab-panel-media-size">${m.sizeStr || 'Streaming...'}</div>
                  <div class="grab-panel-media-url">${truncateUrl(m.url, 50)}</div>
                </div>
                <button class="grab-panel-media-btn ${m.downloadable ? '' : 'disabled'}" ${m.downloadable ? '' : 'disabled'}>
                  ${m.downloadable ? '⬇️' : '📡'}
                </button>
              </div>
            `).join('')}
          </div>
        </div>
      ` : `
        <div class="grab-panel-section">
          <div class="grab-panel-section-title">🎯 Captured Network Requests</div>
          <div class="grab-panel-empty">
            <div class="grab-panel-empty-icon">📡</div>
            <div>No video streams captured yet.</div>
            <div class="grab-panel-empty-hint">Play the video or refresh the page, then reopen this panel.</div>
          </div>
        </div>
      `}
      
      <div class="grab-panel-section">
        <div class="grab-panel-section-title">🎥 Video Elements (${videos.length})</div>
        <div class="grab-panel-videos">
          ${
            videoData.length === 0
              ? `
            <div class="grab-panel-empty">
              <div class="grab-panel-empty-icon">🎬</div>
              No video elements detected
            </div>
          `
              : videoData
                  .map(
                    (v, i) => `
            <div class="grab-panel-video" data-index="${i}">
              <span class="grab-panel-video-icon">🎥</span>
              <div class="grab-panel-video-info">
                <div class="grab-panel-video-type">${v.type.toUpperCase()}</div>
                <div class="grab-panel-video-url">${truncateUrl(v.url, 35) || "No URL"}</div>
              </div>
              <span class="grab-panel-video-status ${v.downloadable ? "downloadable" : v.type}">
                ${v.downloadable ? "OK" : v.type === "blob" ? "BLOB" : v.type === "none" ? "NONE" : "N/A"}
              </span>
            </div>
          `
                  )
                  .join("")
          }
        </div>
      </div>
    </div>
    <div class="grab-panel-footer">
      💡 Captured URLs bypass blob streams!
    </div>
  `;

  document.documentElement.appendChild(panelEl);

  // Close button
  panelEl.querySelector(".grab-panel-close").addEventListener("click", () => {
    panelEl.remove();
    panelEl = null;
  });

  // Captured media download clicks
  panelEl.querySelectorAll(".grab-panel-media-btn:not(.disabled)").forEach((btn) => {
    btn.addEventListener("click", async (e) => {
      e.stopPropagation();
      const idx = parseInt(btn.closest(".grab-panel-media").dataset.capturedIndex, 10);
      const media = capturedMedia[idx];
      
      if (!media || !media.downloadable) return;
      
      btn.textContent = "⏳";
      btn.disabled = true;
      
      const title = sanitizeFilenamePart(document.title) || "video";
      const ts = new Date().toISOString().replace(/[:.]/g, "-").slice(0, 19);
      const ext = guessExtFromUrl(media.url);
      const filename = `GRAB/${title}__${ts}.${ext}`;
      
      try {
        const resp = await new Promise((resolve) => {
          chrome.runtime.sendMessage(
            { type: "GRAB_DOWNLOAD", url: media.url, filename, saveAs: false },
            resolve
          );
        });
        
        if (resp?.ok) {
          btn.textContent = "✅";
          toast("Download started!", "success", "GRAB");
        } else {
          btn.textContent = "❌";
          toast(`Download failed: ${resp?.error || 'Unknown error'}`, "error", "Error");
        }
      } catch (err) {
        btn.textContent = "❌";
        toast(`Error: ${err.message}`, "error");
      }
      
      setTimeout(() => {
        btn.textContent = "⬇️";
        btn.disabled = false;
      }, 2000);
    });
  });

  // Video item clicks
  panelEl.querySelectorAll(".grab-panel-video").forEach((el) => {
    el.addEventListener("click", () => {
      const idx = parseInt(el.dataset.index, 10);
      const { video, downloadable, url, type } = videoData[idx];

      // Scroll video into view
      video.scrollIntoView({ behavior: "smooth", block: "center" });

      // If downloadable, trigger download
      if (downloadable) {
        const btn = overlays.get(video);
        if (btn) btn.click();
      } else {
        toast(
          type === "blob"
            ? "Blob URL - check Captured URLs above for actual stream!"
            : type === "none"
            ? "No source URL found"
            : `${type.toUpperCase()} streaming - check Captured URLs above`,
          "warning"
        );
      }
    });
  });
}

// ===== Message Handler =====
chrome.runtime.onMessage.addListener((msg) => {
  if (msg.type === "GRAB_TOGGLE_PANEL") {
    createPanel();
  }
  
  // Handle real-time media capture notifications
  if (msg.type === "GRAB_MEDIA_CAPTURED") {
    // Update panel if open
    if (panelEl) {
      capturedMedia.push(msg.media);
      // Refresh panel to show new media
      createPanel();
      createPanel(); // Toggle twice to refresh
    }
    
    // Show toast for new captures
    const m = msg.media;
    if (m.downloadable && m.size > 500000) { // > 500KB
      toast(`Captured: ${m.type} ${m.sizeStr}`, "info", "🎯 Video Found!", 3000);
    }
  }
});

// ===== Initialize =====
scanForVideos();

// Watch for dynamically added videos
const mo = new MutationObserver((mutations) => {
  for (const m of mutations) {
    for (const node of m.addedNodes) {
      if (!(node instanceof Element)) continue;
      if (node.tagName === "VIDEO") ensureOverlayForVideo(node);
      else scanForVideos(node);
    }
  }
});
mo.observe(document.documentElement, { childList: true, subtree: true });

console.log("[GRAB] v0.3.1 - Video Download with Duration & Classification initialized");
